//------------------------------------------------------------------------------
//! @file WiMOD_SAP_RadioLink_ProLink_IDs.h
//! @ingroup WiMOD_SAP_RADIOLINK_PROLINK
//! <!------------------------------------------------------------------------->
//! @brief Supporting IDs and definitions for the RadioLink ServiceAccessPoint for LoRaWAN ProLink firmware
//! @version 0.1
//! <!------------------------------------------------------------------------->
//!
//!
//!
//! <!--------------------------------------------------------------------------
//! Copyright (c) 2022
//! IMST GmbH
//! Carl-Friedrich Gauss Str. 2-4
//! 47475 Kamp-Lintfort
//! --------------------------------------------------------------------------->
//! @author (FB), IMST
//! <!--------------------------------------------------------------------------
//! Target OS:    none
//! Target CPU:   tbd
//! Compiler:     tbd
//! --------------------------------------------------------------------------->
//! @internal
//! @par Revision History:
//! <PRE>
//!-----------------------------------------------------------------------------
//! Version | Date       | Author | Comment
//!-----------------------------------------------------------------------------
//!
//! </PRE>
//------------------------------------------------------------------------------


/*
 * THIS IS AN EXAMPLE IMPLEMENTATION ACCORDING THE THE HCI SPEC: V1.3
 * FOR FIRMWARE: LoRaWAN PROLINK
 *
 * SEE FILE: ProLink_LoRaWAN_EndNode_Modem_HCI_Spec_v1_3.pdf for detailed information
 */


#ifndef ARDUINO_SAP_WIMOD_SAP_RADIOLINK_PROLINK_IDS_H_
#define ARDUINO_SAP_WIMOD_SAP_RADIOLINK_PROLINK_IDS_H_

#include "utils/WMDefs.h"

#include "WiMOD_SAP_LORAWAN_IDs_EU868.h"  /* Europe       */
//#include "WiMOD_SAP_LORAWAN_IDs_IN865.h"  /* India        */
#include "WiMOD_SAP_LORAWAN_IDs_US915.h"  /* US           */
#include "WIMOD_SAP_LORAWAN_IDs_AU915.h"  /* Australia    */
#include "WiMOD_SAP_LORAWAN_IDs_AS923.h"  /* Asia/Pacific */
//#include "WiMOD_SAP_LORAWAN_IDs_RU868.h"  /* Russia       */

//------------------------------------------------------------------------------
//
// Service Access Point Identifier
//
//------------------------------------------------------------------------------

/** SAP ID for the RadioLink service */
#define PROLINK_RADIOLINK_SAP_ID                      0x03


//! @cond Doxygen_Suppress

// Status Codes
#define PROLINK_RADIOLINK_STATUS_OK                       0x00 //Operation successful
#define PROLINK_RADIOLINK_STATUS_ERROR                    0x01 //Operation failed
#define PROLINK_RADIOLINK_STATUS_CMD_NOT_SUPPORTED        0x02 //Command is not supported
#define PROLINK_RADIOLINK_STATUS_WRONG_PARAMETER          0x03 //HCI message contains wrong parameter
#define PROLINK_RADIOLINK_STATUS_WRONG_RADIO_MODE         0x04 //Stack is running in a wrong mode
#define PROLINK_RADIOLINK_STATUS_BUFFER_FULL              0x07 //Message queue is full, command rejected
#define PROLINK_RADIOLINK_STATUS_LENGTH_ERROR             0x08 //HCI message length is invalid or radio payload size is too large

//! @endcond




// Message IDs
#define PROLINK_RADIOLINK_MSG_SEND_U_DATA_REQ            0x01
#define PROLINK_RADIOLINK_MSG_SEND_U_DATA_RSP            0x02

#define PROLINK_RADIOLINK_MSG_U_DATA_RX_IND              0x04
#define PROLINK_RADIOLINK_MSG_U_DATA_TX_IND              0x06

#define PROLINK_RADIOLINK_MSG_SET_RADIO_CFG_REQ          0x17
#define PROLINK_RADIOLINK_MSG_SET_RADIO_CFG_RSP          0x18

#define PROLINK_RADIOLINK_MSG_GET_RADIO_CFG_REQ          0x19
#define PROLINK_RADIOLINK_MSG_GET_RADIO_CFG_RSP          0x1A

#define PROLINK_RADIOLINK_MSG_RESET_RADIO_CFG_REQ        0x1B
#define PROLINK_RADIOLINK_MSG_RESET_RADIO_CFG_RSP        0x1C

#define PROLINK_RADIOLINK_MSG_SET_AES_KEY_REQ            0x21
#define PROLINK_RADIOLINK_MSG_SET_AES_KEY_RSP            0x22

#define PROLINK_RADIOLINK_MSG_GET_AES_KEY_REQ            0x23
#define PROLINK_RADIOLINK_MSG_GET_AES_KEY_RSP            0x24



/**
 * @brief general operation mode of the radio part only for LR-BASE mode in ProLink
 */
typedef enum TRadioCfg_RadioModeProLink
{
    ProLinkRadioMode_Standard = 0,                                              /*!< normal operation mode */
    ProLinkRadioMode_Reserved,                                                  /*!< DO NOT USE */
} TRadioCfg_RadioModeProLink;

/**
 * @brief Modulation technique supported by the WiMOD only for LR-BASE mode in ProLink
 */
typedef enum TRadioCfg_ModulationProLink
{
    ProLinkModulation_LoRa = 0,                                                 /*!< LoRa modulation */
    ProLinkModulation_FSK,                                                      /*!< FSK modulation  */
} TRadioCfg_ModulationProLink;


/**
 * @brief Bandwidth settings for the LoRa modulation only for LR-BASE mode in ProLink
 */
typedef enum TRadioCfg_LoRaBandwidthProLink
{
    ProLinkLoRaBandwith_125kHz = 0,                                             /*!< 125 kHz */
    ProLinkLoRaBandwith_250kHz,                                                 /*!< 250 kHz */
    ProLinkLoRaBandwith_500kHz                                                  /*!< 500 kHz */
} TRadioCfg_LoRaBandwidthProLink;

/**
 * @brief SpreadingFactors for LoRa modulation only for LR-BASE mode in ProLink
 */
typedef enum TRadioCfg_LoRaSpreadingFactorProLink
{
    ProLinkLoRa0_SF7 = 0,                                                       /*!< Spreading Factor 7 */
    ProLinkLoRa1_SF7,                                                           /*!< Spreading Factor 7 */
    ProLinkLoRa2_SF7,                                                           /*!< Spreading Factor 7 */
    ProLinkLoRa3_SF7,                                                           /*!< Spreading Factor 7 */
    ProLinkLoRa4_SF7,                                                           /*!< Spreading Factor 7 */
    ProLinkLoRa5_SF7,                                                           /*!< Spreading Factor 7 */
    ProLinkLoRa6_SF7,                                                           /*!< Spreading Factor 7 */
    ProLinkLoRa7_SF7,                                                           /*!< Spreading Factor 7 */
    ProLinkLoRa8_SF8,                                                           /*!< Spreading Factor 8 */
    ProLinkLoRa9_SF9,                                                           /*!< Spreading Factor 9 */
    ProLinkLoRa10_SF10,                                                         /*!< Spreading Factor 10 */
    ProLinkLoRa11_SF11,                                                         /*!< Spreading Factor 11 */
    ProLinkLoRa12_SF12,                                                         /*!< Spreading Factor 12 */
} TRadioCfg_LoRaSpreadingFactorProLink;


/**
 * @brief ErrorCoding settings (for LoRa modulation) only for LR-BASE mode in ProLink
 */
typedef enum TRadioCfg_ErrorCodingProLink
{
    ProLinkErrorCoding0_4_5 = 0,                                                /*!< ErrorCoding 4/5 */
    ProLinkErrorCoding1_4_5,                                                    /*!< ErrorCoding 4/5 */
    ProLinkErrorCoding2_4_6,                                                    /*!< ErrorCoding 4/6 */
    ProLinkErrorCoding3_4_7,                                                    /*!< ErrorCoding 4/7 */
    ProLinkErrorCoding4_4_8,                                                    /*!< ErrorCoding 4/8 */
} TRadioCfg_ErrorCodingProLink;

/**
 * @brief TxPowerLevel settings only for LR-BASE mode in ProLink
 */
typedef enum TRadioCfg_TxPowerLevelProLink
{
    ProLinkTxPowerLevel0_5_dBm = 0,                                             /*!< 5 dBm */
    ProLinkTxPowerLevel1_5_dBm = 1,                                             /*!< 5 dBm */
    ProLinkTxPowerLevel2_5_dBm = 2,                                             /*!< 5 dBm */
    ProLinkTxPowerLevel3_5_dBm = 3,                                             /*!< 5 dBm */
    ProLinkTxPowerLevel4_5_dBm = 4,                                             /*!< 5 dBm */
    ProLinkTxPowerLevel5_5_dBm = 5,                                             /*!< 5 dBm */
    ProLinkTxPowerLevel6_6_dBm,                                                 /*!< 6 dBm */
    ProLinkTxPowerLevel7_7_dBm,                                                 /*!< 7 dBm */
    ProLinkTxPowerLevel8_8_dBm,                                                 /*!< 8 dBm */
    ProLinkTxPowerLevel9_9_dBm,                                                 /*!< 9 dBm */
    ProLinkTxPowerLevel10_10_dBm,                                               /*!< 10 dBm */
    ProLinkTxPowerLevel11_11_dBm,                                               /*!< 11 dBm */
    ProLinkTxPowerLevel12_12_dBm,                                               /*!< 12 dBm */
    ProLinkTxPowerLevel13_13_dBm,                                               /*!< 13 dBm */
    ProLinkTxPowerLevel14_14_dBm,                                               /*!< 14 dBm */
    ProLinkTxPowerLevel15_15_dBm,                                               /*!< 15 dBm */
    ProLinkTxPowerLevel16_16_dBm,                                               /*!< 16 dBm */
    ProLinkTxPowerLevel17_17_dBm,                                               /*!< 17 dBm */
    ProLinkTxPowerLevel18_18_dBm,                                               /*!< 18 dBm */
    ProLinkTxPowerLevel19_19_dBm,                                               /*!< 19 dBm */
    ProLinkTxPowerLevel20_20_dBm,                                               /*!< 20 dBm */
} TRadioCfg_TxPowerLevelProLink;


/**
 * @brief RX Operation modes only for LR-BASE mode in ProLink
 */
typedef enum TRadioCfg_RxControlProLink
{
    ProLinkRxCtrl_Receiver_Off = 0,                                             /*!< Disable the RX operation completely */
    ProLinkRxCtrl_Receiver_AlwaysOn,                                            /*!< RX operation is always ON*/
    ProLinkRxCtrl_Receiver_RxWindowed,                                          /*!< RX operation is only used in defined time windows */
} TRadioCfg_RxControlProLink;


// bit field for Misc.Options in RadioConfig structure
#define PROLINK_RADIOLINK_RADIO_CFG_MISC_EXTENDED_HCI_OUTPUT_FORMAT   (1 << 0)
#define PROLINK_RADIOLINK_RADIO_CFG_MISC_HCI_TX_IND_ENABLED           (1 << 2)
#define PROLINK_RADIOLINK_RADIO_CFG_MISC_HCI_ENCTRYPTION_ENABLED      (1 << 5)

//! @endcond

/**
 * @brief Settings for the FSK data rate only for LR-BASE mode in ProLink
 */
typedef enum TRadioCfg_FskDatarateProLink
{
    ProLinkFskDatarate_50kbps = 0,                                              /*!< FSK modulation using 50  kbps */
} TRadioCfg_FskDatarateProLink;

//------------------------------------------------------------------------------

/**
 * @brief Structure containing the radio (stack) config parameters of the proprietary LR-BASE stack (only for ProLink)
 */
typedef struct TWiMODLR_ProLinkRadioLink_RadioConfig
{
    // only for reading
    UINT8                           Status;                                     /*!< Status Flag of HCI get operation; only vaild if config has been requested*/

    // only for writing
    UINT8                           StoreNwmFlag;                               /*!< Store the config in NVM or RAM; 1 = store in NVM*/

    TRadioCfg_RadioModeProLink      RadioMode;                                  /*!< Radio Mode setting */
    UINT8                           GroupAddress;                               /*!< Group address of WiMOD (for RX)*/

    UINT16                          DeviceAddress;                              /*!< RF device address used for RX*/

    TRadioCfg_ModulationProLink     Modulation;                                 /*!< RF Modulation to be used for TX*/
    UINT8                           RfFreq_LSB;                                 /*!< lower part of the 24bit frequency register of the transceiver*/
    UINT8                           RfFreq_MID;                                 /*!< mid part of the 24bit frequency register of the transceiver*/
    UINT8                           RfFreq_MSB;                                 /*!< high part of the 24bit frequency register of the transceiver*/
    TRadioCfg_LoRaBandwidthProLink  LoRaBandWidth;                              /*!< Bandwidth setting for the LoRa modulation */
    TRadioCfg_LoRaSpreadingFactorProLink   LoRaSpreadingFactor;                        /*!< LoRa Spreading factor*/
    TRadioCfg_ErrorCodingProLink    ErrorCoding;                                /*!< Error Coding */
    TRadioCfg_TxPowerLevelProLink   PowerLevel;                                 /*!< Power Level setting*/

    TRadioCfg_RxControlProLink      RxControl;                                  /*!< Options related to RX operations*/
    UINT16                          RxWindowTime;                               /*!< Time settings related to the RX Window */

    UINT8                           MiscOptions;                                /*!< misc options field: see PROLINK_RADIOLINK_RADIO_CFG_MISC defines */
    TRadioCfg_FskDatarateProLink    FskDatarate;                                /*!< Datarate setting for the FSK modulation*/

} TWiMODLR_ProLinkRadioLink_RadioConfig;


//------------------------------------------------------------------------------
//
// misc. defines
//
//------------------------------------------------------------------------------
//! @cond Doxygen_Suppress

#define PROLINK_RADIOLINK_AES_KEY_LEN                         16      // 128 bit

//! @endcond


#endif /* ARDUINO_SAP_WIMOD_SAP_RADIOLINK_PROLINK_IDS_H_ */
